//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.2

import wesual.Controls 1.0

FocusScope {
    id : tabs

    property list<TabItem> items
    property int           margins     : 2
    property int           activeIndex : 0 // first if not defined outside

    property bool          disbaleUpperCase : false
    property bool          centerTabs  : false

    property alias         count       : repeater.count
    property alias         spacing     : tabRow.spacing

    signal indexSelected(var index)

    function tabAtIndex(index) {
        return repeater.itemAt(index);
    }

    // selects tab and triggers action bound to tab
    function selectTabAtIndex(index) {
        if (index < 0 || index > (repeater.model.length - 1))
            return;

        indexSelected(index);
        tabs.activeIndex = index;
        repeater.itemAt(index).selectTab();
        repeater.itemAt(index).focus = true;
    }

    // sets the highlight to the tab with index 'index', the action bound to the
    // tab is not triggered
    function setHighlightToIndex(index) {
        if (index < 0 || index > (repeater.model.length - 1))
            return;

        indexSelected(index);
        tabs.activeIndex = index;
        repeater.itemAt(index).focus = true;
    }

    onIndexSelected : {
        bottomBar.repositioning = true;
        bottomBar.belowTab = tabAtIndex(index);
        bottomBar.x = tabs.mapFromItem(bottomBar.belowTab, 0, 0).x;
    }

    // react to external tab highlight
    onActiveIndexChanged : {
        indexSelected(activeIndex);
        //repeater.itemAt(activeIndex).selectTab();
        repeater.itemAt(activeIndex).focus = true;
    }

    activeFocusOnTab : true
    onActiveFocusChanged : {
        if (activeFocus && activeIndex !== -1)
            repeater.itemAt(activeIndex).focus = true;
    }

    Component.onCompleted : {
        indexSelected(activeIndex); // position tab underline
    }

    focus : true
    height : 35
    implicitWidth : childrenRect.width
    onWidthChanged : { // update unline position when rearanging
        tabRow.tabRowWidth = Math.max(tabRow.implicitWidth, tabRow.tabRowWidth);
        bottomBar.repositioning = false;
        bottomBar.x = tabs.mapFromItem(bottomBar.belowTab, 0, 0).x;
    }

    Rectangle {
        id : bottomSeparator
        height : 1
        width : parent.width
        anchors {
            left  : parent.left
            right : parent.right
            bottom : parent.bottom
        }
        color : UI.color(UI.TertiarySeparationLine)
    }

    Row {
        id : tabRow

        spacing : 18

        // prevent "tab item jumping" on repeater redraw
        property int tabRowWidth : 0
        property int maxItemWidth : tabs.width
                                    ? (tabs.width-(tabs.count-1)*spacing
                                       -tabs.count*2*tabs.margins)/tabs.count
                                    : 50
        onMaxItemWidthChanged : {
            // apply correct tab available text widths
            for (var i = 0; i < repeater.count; i++) {
                var tab = repeater.itemAt(i);

                if (tabRowWidth > tabs.width) {
                    if (maxItemWidth > 10 && tab.implicitWidth > maxItemWidth) {
                        tab.text.width = maxItemWidth;
                    } else if (maxItemWidth <= 10) {
                        tab.text.width = 10;
                    } else if (tab.text.width < maxItemWidth
                               || tab.implicitWidth > tab.text.width) {
                        tab.text.width = maxItemWidth;
                    }
                } else {
                    tab.text.width = tab.text.implicitWidth + 2*tabs.margins;
                }
            }
        }
        height : childrenRect.height
        anchors.verticalCenter : parent.verticalCenter
        anchors.horizontalCenter : {
            if (tabs.centerTabs)
                return parent.horizontalCenter;
        }

        Repeater {
            id : repeater
            model : items

            delegate : Item {
                id : delegate

                property bool  active       : tabs.activeIndex === index
                property alias mouse        : mouseArea
                property alias text         : content

                height : content.implicitHeight + 2*tabs.margins
                implicitWidth : content.contentWidth + 2*tabs.margins

                Rectangle {
                    id : focusBorder

                    width : parent.width
                    height : parent.height
                    anchors.verticalCenter : parent.verticalCenter
                    anchors.verticalCenterOffset : -2
                    border {
                        width : 1
                        color : UI.color(UI.PrimaryFocusBorder, 0.4)
                    }
                    color : UI.color(UI.PrimaryFocusBorder, 0.2)
                    visible : parent.activeFocus
                }

                Text {
                    id : content
                    font {
                        family : UI.fontFamily(UI.SecondaryFont)
                        weight : UI.fontWeight(UI.SecondaryFont)
                        pixelSize : 13
                        capitalization : tabs.disbaleUpperCase
                                         ? Font.Normal : Font.AllUppercase
                    }
                    maximumLineCount : 1
                    text : action.text
                    elide : Qt.ElideRight
                    color : {
                        if (!action.enabled)
                            return UI.color(UI.DisabledTabText);
                        if (delegate.active)
                            return UI.color(UI.SecondaryBase);
                        if (mouseArea.pressed)
                            return UI.color(UI.SecondaryPress);
                        if (mouseArea.containsMouse)
                            return UI.color(UI.SecondaryHover);

                        return UI.color(UI.SecondaryTextColor);
                    }

                    anchors {
                        left : parent.left
                        leftMargin : 2
                        top : parent.top
                    }
                    height : 23

                    Behavior on color { ColorAnimation { duration : 0 } }
                }

                MouseArea {
                    id : mouseArea
                    anchors.fill : delegate
                    hoverEnabled : true
                    enabled : action.enabled
                    acceptedButtons : Qt.LeftButton
                    onClicked : {
                        tabs.forceActiveFocus();
                        selectTab();
                    }
                    cursorShape : action.enabled ? Qt.PointingHandCursor :
                                                   Qt.ArrowCursor
                }

                // key handling
                Keys.onLeftPressed : {
                    var nextItem = repeater.itemAt(Math.max(index - 1, 0));

                    if (nextItem.mouse.enabled)
                        nextItem.focus = true;
                }
                Keys.onRightPressed : {
                    var nextIndex = Math.min(index + 1,
                                             repeater.model.length - 1);
                    var nextItem  = repeater.itemAt(nextIndex);

                    if (nextItem.mouse.enabled)
                        nextItem.focus = true;
                }
                Keys.onEnterPressed  : selectTab()
                Keys.onReturnPressed : selectTab()

                function selectTab() {
                    if (delegate.active)
                        return; // don't fire on reselection

                    tabs.activeIndex = index;
                    modelData.selected();
                    if (action)
                        action.trigger();

                    tabs.indexSelected(index);
                }
            } // delegate
        } // repeater
    } // tabRow

    Rectangle {
        id : bottomBar

        property Item belowTab : null
        property bool repositioning : false
        width : belowTab.width
        Behavior on width {
            enabled : bottomBar.repositioning
            NumberAnimation { duration : 200 }
        }

        height : 1

        x : tabs.mapFromItem(belowTab, 0, 0).x
        Behavior on x {
            enabled : bottomBar.repositioning
            NumberAnimation { duration : 200 }
        }
        y : tabs.height - height

        color : {
            if(belowTab.mouse)
                return UI.color(UI.SecondaryHover);

            return UI.color(UI.TabButtonBorder);
        }
        Behavior on color { ColorAnimation { duration : 0 } }

        states : State {
            name : "init"
            when : bottomBar.belowTab !== null
            PropertyChanges {
                target : bottomBar
                x : tabs.mapFromItem(bottomBar.belowTab, 0, 0).x
            }
        }
        transitions : Transition {
            PropertyAnimation {
                properties : "x"
                duration : 200
            }
        }
    } // bottom tab underline
}

